from django.shortcuts import render, redirect
from django.contrib.auth.decorators import login_required
from django.contrib import messages
from django.contrib.auth import authenticate, login, logout
from .forms import CustomerLocationForm
from .models import CustomerLocation
from django.core.exceptions import RequestDataTooBig, ValidationError
from django.conf import settings
from django.db import IntegrityError
from django.http import JsonResponse
from django.core.paginator import Paginator
from django.db.models import Q

# Create your views here.

def login_view(request):
    if request.user.is_authenticated:
        return redirect('main:add_customer_location')
        
    if request.method == 'POST':
        username = request.POST.get('username')
        password = request.POST.get('password')
        user = authenticate(request, username=username, password=password)
        
        if user is not None:
            login(request, user)
            messages.success(request, 'Đăng nhập thành công!')
            return redirect('main:add_customer_location')
        else:
            messages.error(request, 'Tên đăng nhập hoặc mật khẩu không đúng!')
    
    return render(request, 'main/login.html')

def logout_view(request):
    logout(request)
    messages.info(request, 'Đã đăng xuất thành công!')
    return redirect('main:login')

@login_required(login_url='main:login')
def add_customer_location(request):
    try:
        if request.method == 'POST':
            # Kiểm tra kích thước file
            if request.META.get('CONTENT_LENGTH'):
                if int(request.META['CONTENT_LENGTH']) > settings.UPLOAD_SIZE_LIMIT:
                    messages.error(request, 'File quá lớn. Kích thước tối đa cho phép là 10MB.')
                    return redirect('main:add_customer_location')

            form = CustomerLocationForm(request.POST)
            if form.is_valid():
                try:
                    # Kiểm tra số hợp đồng trùng lặp
                    contract_number = form.cleaned_data['contract_number']
                    if CustomerLocation.objects.filter(contract_number=contract_number).exists() and not request.POST.get('confirm_duplicate'):
                        return render(request, 'main/add_customer_location.html', {
                            'form': form,
                            'show_confirm': True,
                            'contract_number': contract_number
                        })

                    customer_location = form.save(commit=False)
                    customer_location.user = request.user
                    customer_location.acct_ktv = request.user.username
                    
                    # Validate dữ liệu
                    if not request.POST.get('latitude') or not request.POST.get('longitude'):
                        messages.error(request, 'Vui lòng cho phép truy cập vị trí để lấy tọa độ.')
                        return render(request, 'main/add_customer_location.html', {'form': form})
                    
                    if not request.POST.get('image_urls'):
                        messages.error(request, 'Vui lòng tải lên ít nhất một hình ảnh.')
                        return render(request, 'main/add_customer_location.html', {'form': form})

                    customer_location.image_urls = request.POST.get('image_urls')
                    customer_location.latitude = request.POST.get('latitude')
                    customer_location.longitude = request.POST.get('longitude')
                    customer_location.save()
                    messages.success(request, 'Thông tin đã được lưu thành công!')
                    return redirect('main:add_customer_location')

                except IntegrityError:
                    messages.error(request, 'Số hợp đồng đã tồn tại trong hệ thống.')
                except ValidationError as e:
                    messages.error(request, f'Lỗi dữ liệu: {str(e)}')
                except Exception as e:
                    messages.error(request, f'Có lỗi xảy ra: {str(e)}')
            else:
                for field, errors in form.errors.items():
                    for error in errors:
                        messages.error(request, f'Lỗi ở trường {field}: {error}')
        else:
            form = CustomerLocationForm(initial={'acct_ktv': request.user.username})
        
        return render(request, 'main/add_customer_location.html', {'form': form})

    except RequestDataTooBig:
        messages.error(request, 'Dữ liệu gửi lên quá lớn. Vui lòng kiểm tra kích thước file.')
        return redirect('main:add_customer_location')
    except Exception as e:
        messages.error(request, f'Có lỗi không mong muốn xảy ra: {str(e)}')
        return redirect('main:add_customer_location')

@login_required(login_url='main:login')
def search_locations(request):
    query = request.GET.get('contract_number', '')
    page = request.GET.get('page', 1)
    
    # Xử lý tìm kiếm
    if query:
        locations = CustomerLocation.objects.filter(
            Q(contract_number__icontains=query)
        )
    else:
        locations = None
    
    # Lấy lịch sử và phân trang
    all_locations = CustomerLocation.objects.all().order_by('-created_at')
    paginator = Paginator(all_locations, 3)  # 10 items mỗi trang
    history = paginator.get_page(page)
    
    context = {
        'locations': locations,
        'query': query,
        'history': history,
    }
    
    return render(request, 'main/search_locations.html', context)

@login_required(login_url='main:login')
def user_guide(request):
    return render(request, 'main/user_guide.html')

@login_required(login_url='main:login')
def check_contract_number(request):
    contract_number = request.GET.get('contract_number', '')
    exists = CustomerLocation.objects.filter(contract_number=contract_number).exists()
    return JsonResponse({'exists': exists})
